DROP PROCEDURE [hips].[PatientMasterGet]
GO
SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON
GO
-- =============================================
-- Author:        Andrew Jacks
-- Create date: 12 October 2012
-- Description:   Gets a specific record from the Patient Master table.
--
-- Lookup Types Supported:
--    1. PatientMasterId (e.g. 1234567)
--    2. StatePatientId (e.g. '100000090751')
--    3. Ihi (e.g. '8003604567901177')
--    4. HospitalId and Mrn  (e.g. 71, '01234567')
--    5. HospitalCode and CodeSystemId and Mrn ('FMC', 2, '01234567')
--    6. Demographic search using all of:
--         CurrentLastName, CurrentFirstNames, DateOfBirth, CurrentSexId
--       and one of:
--           DvaNumber, MedicareNumber
--       and optionally MedicareIrn.
--       For demographic search the business logic must pass a valid medicare
--       number or null, but avoid passing a special value like '8' or '0',
--       which could match records that it should not match.
--
-- Modified: 23/11/2012 sbiber added IHI and StatePatientId lookup
-- Modified: 18/12/2012 sbiber removed PcehrAdvertised
-- Modified: 24/01/2012 sbiber added conditions to prevent returning all patients
-- Modified: 07/05/2013 sbiber tightened use of demographic search
-- Modified: 16/06/2013 pstall01 added date of death indicator
-- =============================================
CREATE PROCEDURE [hips].[PatientMasterGet] 
(
	@PatientMasterId INT = NULL,
	@HospitalId INT = NULL,
	@HospitalCode VARCHAR(64) = NULL,
	@CodeSystemId INT = NULL,
	@Mrn VARCHAR(20) = NULL,
	@CurrentLastName VARCHAR(80) = NULL,
	@CurrentFirstNames VARCHAR(80) = NULL,
	@DateOfBirth DATETIME = NULL,
	@CurrentSexId INT = NULL,
	@MedicareNumber VARCHAR(12) = NULL,
	@MedicareIrn VARCHAR(3) = NULL,
	@DvaNumber VARCHAR(18) = NULL,
	@Ihi VARCHAR(16) = NULL,
	@StatePatientId VARCHAR(64) = NULL
)
AS
BEGIN
	IF @HospitalId IS NULL AND @CodeSystemId IS NOT NULL AND @HospitalCode IS NOT NULL 
	BEGIN
	  SELECT @HospitalId = [HospitalId]
	  FROM [hips].[HospitalCode]
	  WHERE [CodeSystemId] = @CodeSystemId
	  AND [Code] = @HospitalCode
	END


	IF @PatientMasterID IS NULL
	BEGIN
		IF @HospitalId IS NOT NULL AND @Mrn IS NOT NULL
		BEGIN
			SELECT	@PatientMasterId = [PatientMasterId]
			FROM	[hips].[HospitalPatient]
			WHERE	[HospitalId] = @HospitalId
			AND		[Mrn] = @Mrn
		END
		ELSE IF @Ihi IS NOT NULL
		BEGIN
			SELECT	@PatientMasterId = [PatientMasterId]
			FROM	[hips].[PatientMasterIhi]
			WHERE	[Ihi] = @Ihi
		END
		ELSE IF @StatePatientId IS NOT NULL
		BEGIN
			SELECT	@PatientMasterId = [PatientMasterId]
			FROM	[hips].[PatientMaster]
			WHERE	[StatePatientId] = @StatePatientId
		END				
	END	

	IF  @PatientMasterId IS NULL
	AND @Mrn IS NULL
	AND @StatePatientId IS NULL
	AND @Ihi IS NULL
	AND @CurrentFirstNames IS NOT NULL 
	AND @CurrentLastName IS NOT NULL
	AND @DateOfBirth IS NOT NULL
	AND (  @MedicareNumber IS NOT NULL
		OR @DvaNumber IS NOT NULL
		)
	BEGIN
		SELECT pm.[PatientMasterId]
			 , COALESCE (pmi.RegisteredSexId,pm.CurrentSexId, -1) AS RegisteredSexId
			 , COALESCE (pm.CurrentSexId, -1) AS CurrentSexId
			 , pmi.Ihi
			 , COALESCE (pmi.IhiStatusId, -1) AS IhiStatusId
			 , ihis.Description AS IhiStatus
			 , COALESCE (pmi.IhiRecordStatusId, -1) AS IhiRecordStatusId
			 , pmi.[RegisteredFamilyName]
			 , pmi.[RegisteredGivenName]
			 , irs.Description AS IhiRecordStatus
			 , pmi.DateLastValidated AS IhiLastValidated
			 , pm.[MedicareNumber]
			 , pm.[MedicareIrn]
			 , pm.[IsMedicareNumberValid]
			 , pm.[DvaNumber]
			 , pm.[StatePatientId]
			 , pm.[DateOfBirth]
			 , pm.[DateOfDeath]
			 , pm.[DeathIndicatorId]
			 , pm.[DateCreated]
			 , pm.[UserCreated]
			 , pm.[DateModified]
			 , pm.[UserModified]
        FROM	[hips].[PatientMaster] AS pm
        LEFT JOIN [hips].[PatientMasterIhi] AS pmi ON pmi.[PatientMasterID] = pm.[PatientMasterID]
        LEFT JOIN [hips].HospitalPatient AS p ON pm.PatientMasterID = p.PatientMasterID
		LEFT JOIN [hips].[Hospital] AS h ON h.HospitalID = p.HospitalID
		LEFT JOIN [hips].PatientMasterName pmn ON pmn.PatientMasterId = pm.PatientMasterId AND pmn.NameTypeId = 2
		LEFT JOIN [hips].IhiRecordStatus AS irs on irs.IhiRecordStatusId = pmi.IhiRecordStatusId
		LEFT JOIN [hips].IhiStatus AS ihis on ihis.IhiStatusId = pmi.IhiStatusId
		
        WHERE	(pmn.[FamilyName] = @CurrentLastName)
		AND		(pmn.[GivenNames] = @CurrentFirstNames)
		AND		(pm.[DateOfBirth] = @DateOfBirth)
		AND		(pm.[CurrentSexID] = @CurrentSexID)
		AND		(@DvaNumber IS NULL OR pm.[DvaNumber] = @DvaNumber)
		-- The business logic must ensure that special values (invalid/ineligible) are passed in as null to avoid matching
		AND		(@MedicareNumber IS NULL OR pm.[MedicareNumber] = @MedicareNumber)

		-- Do match even if the record has a null sequence number
		AND		(@MedicareIrn IS NULL OR pm.[MedicareIrn] IS NULL OR pm.[MedicareIrn] = @MedicareIrn)
	END
	ELSE  -- Lookup by @PatientMasterID
	BEGIN
		SELECT pm.[PatientMasterID]
			 , COALESCE (pmi.RegisteredSexId,pm.CurrentSexId, -1) AS RegisteredSexId
			 , COALESCE (pm.CurrentSexId, -1) AS CurrentSexId
			 , pmi.Ihi
			 , COALESCE (pmi.IhiStatusId, -1) AS IhiStatusId
			 , ihis.Description AS IhiStatus
			 , COALESCE (pmi.IhiRecordStatusId, -1) AS IhiRecordStatusId
			 , irs.Description AS IhiRecordStatus
			 , pmi.[RegisteredFamilyName]
			 , pmi.[RegisteredGivenName]
			 , pmi.DateLastValidated AS IhiLastValidated
			 , pm.[MedicareNumber]
			 , pm.[MedicareIrn]
			 , pm.[IsMedicareNumberValid]
			 , pm.[DvaNumber]
			 , pm.[StatePatientId]
			 , pm.[DateOfBirth]
			 , pm.[DateOfDeath]
			 , pm.[DeathIndicatorId]
			 , pm.[DateCreated]
			 , pm.[UserCreated]
			 , pm.[DateModified]
			 , pm.[UserModified]         
	     FROM	[hips].[PatientMaster] AS pm
         LEFT JOIN [hips].[PatientMasterIhi] AS pmi ON pmi.[PatientMasterID] = pm.[PatientMasterID]
         LEFT JOIN [hips].IhiRecordStatus AS irs on irs.IhiRecordStatusId = pmi.IhiRecordStatusId
		 LEFT JOIN [hips].IhiStatus AS ihis on ihis.IhiStatusId = pmi.IhiStatusId

        WHERE	pm.[PatientMasterID] = @PatientMasterID
		AND     (@DateOfBirth IS NULL OR @DateOfBirth = [DateOfBirth])
	END
END
GO

